<?php
/**
 * Plugin Name: Dogri Elementor
 * Description: Create Unlimited Widgets with Elementor Page Builder.
 * Plugin URI:  https://themeforest.net/user/gramentheme/portfolio
 * Version:     1.0.0
 * Author:      Gramentheme
 * Author URI:  https://themeforest.net/user/gramentheme/portfolio
 * Text Domain: dogri-elementor
 * Domain Path: /languages/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Dogri Elementor Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class DogriElementor {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var DogriElementor The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return DogriElementor An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {

		add_action( 'init', [ $this, 'i18n' ] );
		add_action( 'plugins_loaded', [ $this, 'init' ] );

	}

	/**
	 * Load Textdomain
	 *
	 * Load plugin localization files.
	 *
	 * Fired by `init` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function i18n() {

		load_plugin_textdomain( 'dogri-elementor' );

	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}

		add_action( 'elementor/init', [ $this, 'add_elementor_category' ], 1 );

		// Add Plugin actions
		add_action( 'elementor/frontend/after_register_scripts', [ $this, 'register_frontend_scripts' ], 10 );

		// Register Widget Styles
		add_action( 'elementor/frontend/after_enqueue_styles', [ $this, 'register_frontend_styles' ] );

		// update code for elementor 3.16v
		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );

		// Register controls
		add_action( 'elementor/controls/register', [ $this, '_register_controls' ] );
	}


	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'dogri-elementor' ),
			'<strong>' . esc_html__( 'Dogri Elementor', 'dogri-elementor' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'dogri-elementor' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'dogri-elementor' ),
			'<strong>' . esc_html__( 'Dogri Elementor', 'dogri-elementor' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'dogri-elementor' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'dogri-elementor' ),
			'<strong>' . esc_html__( 'Dogri Elementor', 'dogri-elementor' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'dogri-elementor' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Add Elementor category.
	 */
	public function add_elementor_category() {
    	\Elementor\Plugin::instance()->elements_manager->add_category('dogri-elementor',
	      	array(
					'title' => __( 'Dogri Elementor', 'dogri-elementor' ),
					'icon'  => 'fa fa-plug',
	      	) 
	    );
	}

	/**
	* Register Frontend Scripts
	*
	*/
	public function register_frontend_scripts() {
		wp_register_script( 'dogri-elementor', plugin_dir_url( __FILE__ ) . 'assets/js/dogri-elementor.js', array( 'jquery' ), self::VERSION );
	}

	/**
	* Register Frontend styles
	*
	*/
	public function register_frontend_styles() {
		wp_register_style( 'dogri-elementor', plugin_dir_url( __FILE__ ) . 'assets/css/dogri-elementor.css', self::VERSION );
	}




	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets() {

		// Include Widget files

		// require_once plugin_dir_path( __FILE__ ) . 'customicon/icon.php';

		require_once plugin_dir_path( __FILE__ ) . 'widgets/banner_widget.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/banner_2_widget.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/feature_widget.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/about_widget.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/service_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/appointment_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/choose_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/brand_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/testimonial_slider.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/skill_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/team_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/appointment_form.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/blog_post_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/about_with_progress_bar.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/service_big_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/benefit_tab.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/team_2_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/intro_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/blog_post_list.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/newsletter_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/banner_3_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/about_with_images.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/counter_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/service_2_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/appointment_form_2.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/choose_slider.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/review_slider.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/beforeafter_widget.php';	
		require_once plugin_dir_path( __FILE__ ) . 'widgets/about_2_widget.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/work_process.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/service_details.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/doctor_team_widget.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/doctor_details.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/skill_progress_bar.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/project_widget.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/contact_box.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/banner_slider.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/about_tab.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/banner_4_widget.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/service_with_icon.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/dogri_service_post_widget.php';		
		require_once plugin_dir_path( __FILE__ ) . 'widgets/portfolio_post_widget.php';		


		// Register widget
		
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Banner_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Banner_2_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Feature_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\About_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Service_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Appointment_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Choose_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Brand_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Testimonial_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Skill_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Team_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Appointment_Form() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Blog_Post_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\About_With_Progress_Bar() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Service_Big_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Benefit_Tab() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Team_2_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Intro_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Blog_Post_List() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Newsletter_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Banner_3_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\About_With_Images() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Counter_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Service_2_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Appointment_Form_2() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Choose_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Review_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Beforeafter_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\About_2_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Work_Process() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Service_Details() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Doctor_Team_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Doctor_Details() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Skill_Progress_Bar() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Project_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Contact_Box() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Banner_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\About_Tab() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Banner_4_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Service_With_Icon() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Dogri_Service_Post_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \DogriElementor\Widget\Portfolio_Post_Widget() );
		


	}

	/** 
	 * _register_controls description
	 * @return [type] [description]
	 */
	public function _register_controls() {

		$controls_manager = \Elementor\Plugin::$instance->controls_manager;
		// $controls_manager->register_control( 'slider-widget', new Test_Control1() );
	
	}

	/**
	 * Prints the Elementor Page content.
	 */
	public static function get_content( $id = 0 ) {
		if ( class_exists( '\ElementorPro\Plugin' ) ) {
			echo do_shortcode( '[elementor-template id="' . $id . '"]' );
		} else {
			echo \Elementor\Plugin::instance()->frontend->get_builder_content_for_display( $id );
		}
	}

}

DogriElementor::instance();
